/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/ 

#include "DM814x_EVM.h"
#include "emac.h"
#include "stdio.h"
#include "string.h"


#define TX_BUF    128
#define RX_BUF    128

static UINT8 packet_data[TX_BUF];

static UINT8 packet_buffer1[RX_BUF];
static UINT8 packet_buffer2[RX_BUF];

UINT16 gmii_phy_getReg( INT16 phynum, INT16 regnum );

/*
 *  We use pDescBase for a base address. Its easier this way
 *  because we can use indexing off the pointer.
 */
static EMAC_Desc* pDescBase = ( EMAC_Desc* )EMAC_RAM_BASE;

/*
 *  The following are used to allow the ISR to communicate with
 *  the application.
 */
 volatile INT32 RxCount;
 volatile INT32 TxCount;
 volatile INT32 ErrCount;
 volatile EMAC_Desc *pDescRx;
 volatile EMAC_Desc *pDescTx;

/* ------------------------------------------------------------------------ *
 *  gmii_phy_detect( *phyaddr )                                             *
 * ------------------------------------------------------------------------ */
UINT16 gmii_phy_detect( UINT16 *phyaddr )
{
	UINT16 num = 0, i;

	/* Reset Ethernet */
    EMAC_SOFTRESET = 1;
    while( EMAC_SOFTRESET != 0 );
    DM814x_usecDelay( 100 );
    
    
    MDIO_CONTROL = 0x40000020;              // Enable MII interface ( MDIOCLK < 12.5MHz )
    DM814x_usecDelay( 100000 );
    /* Check for PHYs */

  	for(i=0;i<32;i++)
  	{
   	    if((MDIO_ALIVE & (1 << i)) != 0)
   	    {
   	        printf("    PHY found at address %d\n", i);
   	        *phyaddr++ = i;
   	    }
  	}
    return num;
}

/* ------------------------------------------------------------------------ *
 *  gmii_phy_getReg( phynum, regnum )                                       *
 * ------------------------------------------------------------------------ */
UINT16 gmii_phy_getReg( INT16 phynum, INT16 regnum )
{
    UINT16 value;

    MDIO_USERACCESS0 = 0                    // Read Phy Id 1
        | ( 1 << 31 )                       // [31] Go
        | ( 0 << 30 )                       // [30] Read
        | ( 0 << 29 )                       // [29] Ack
        | ( regnum << 21 )                  // [25-21] PHY register address
        | ( phynum << 16 )                  // [20-16] PHY address
        | ( 0 << 0 );                       // [15-0] Data

    while( MDIO_USERACCESS0 & 0x80000000 ); // Wait for Results

    value = MDIO_USERACCESS0;
    return value;
}

/* ------------------------------------------------------------------------ *
 *  gmii_phy_setReg( phynum, regnum, data )                                 *
 * ------------------------------------------------------------------------ */
void gmii_phy_setReg( INT16 phynum, INT16 regnum, UINT16 data )
{
    MDIO_USERACCESS0 = 0                    // Read Phy Id 1
        | ( 1 << 31 )                       // [31] Go
        | ( 1 << 30 )                       // [30] Write
        | ( 0 << 29 )                       // [29] Ack
        | ( regnum << 21 )                  // [25-21] PHY register address
        | ( phynum << 16 )                  // [20-16] PHY address
        | ( data << 0 );                    // [15-0] Data

    while( MDIO_USERACCESS0 & 0x80000000 ); // Wait for Results
    DM814x_usecDelay( 1000 );
}

/* ------------------------------------------------------------------------ *
 *  gmii_phy_dumpRegs( )                                                    *
 * ------------------------------------------------------------------------ */
void gmii_phy_dumpRegs( )
{
    INT16 i;
    for ( i = 0 ; i < 32 ; i++ )
        printf( "PHY[%d] = %04x\n", i, gmii_phy_getReg( 1, i ) );

    printf( "\n\n" );
}

/* ------------------------------------------------------------------------ *
 *  emac_gmii_init( phynum )                                                       *
 * ------------------------------------------------------------------------ */
INT16 emac_gmii_init( INT16 phynum )
{
    INT16 i;
    volatile UINT32* pReg;
    INT16 readData = 0;
//  UINT32 ale_entry[3] = {0,0,0};

	/* 
	 * EMAC Boot code sets this offset to 0x21 
	 * This initializes EMAC to work in all boot settings 
	 */
	EMAC0_SLIVER_CTRL = 0x000000A1;
    EMAC1_SLIVER_CTRL = 0x000000A1;

    /* Reset Ethernet */
    EMAC_SOFTRESET = 1;
    while( EMAC_SOFTRESET != 0 );
    DM814x_usecDelay( 100 );


	EMAC_ALE_CONTROL |= 0x80000000;
	DM814x_usecDelay( 100000 ); 
	EMAC_ALE_CONTROL |= 0x40000000;
	DM814x_usecDelay( 100000 );
	DM814x_usecDelay( 100000 ); 
	EMAC_ALE_CONTROL |= 0x00000010;
	DM814x_usecDelay( 100000 ); 

    /* ---------------------------------------------------------------- *
     *  Init PHY / MDIO                                                 *
     * ---------------------------------------------------------------- */
    MDIO_CONTROL = 0x40000020;              // Enable MII interface ( MDIOCLK < 12.5MHz )
    DM814x_usecDelay( 100000 );

    gmii_phy_setReg( phynum,  0x1D, 0x0b);
    readData = gmii_phy_getReg( phynum, 0x1E );
    readData &= ~(0x8000);
    gmii_phy_setReg( phynum,  0x1D, 0x0b);
    gmii_phy_setReg( phynum,  0x1E, readData);

    gmii_phy_setReg( phynum,  0x1D, 0x11);
    readData = gmii_phy_getReg( phynum, 0x1E );
    readData |= 0x01;
    gmii_phy_setReg( phynum,  0x1D, 0x11);
    gmii_phy_setReg( phynum,  0x1E, readData);

    gmii_phy_setReg( phynum,  0, 0x8140);       // Enable external loopback with 1000; Full-Duplex
    DM814x_usecDelay( 100000 );

	readData = gmii_phy_getReg( phynum, 0 );
	if (readData == 0x3100)
		printf (" MII 100 Mbps Link.\r\n");
	else if (readData == 0x1140)
		printf ("RGMII 1 Gbps Link.\r\n");
	else if (readData == 0x1100)
		printf ("MII 10 Mbps Link.\r\n");
    
    //Setup GIG advertisement
    if (readData == 0x1140)
    {
		readData = gmii_phy_getReg( phynum, 9 );
		readData |= 0x200;
		gmii_phy_setReg( phynum, 9, readData);
    }
    
    // setup the general advertisement
    readData = gmii_phy_getReg( phynum, 4 );
    readData |= 0x01E0;
    gmii_phy_setReg( phynum, 4, readData);
    
    DM814x_usecDelay( 100000 );
    DM814x_usecDelay( 100000 );

    printf( "    In RGMII mode\n" );

    MDIO_CONTROL = 0x40000020;              // Enable MII interface ( MDIOCLK < 12.5MHz )
	
	EMAC_STAT_PORT_EN = 0x7;
	EMAC_P2_MAX_BLKS = 0x42;
	
    /* Gigabit PHY external cable loopback */
    gmii_phy_setReg( phynum, 19, 0x0080 );       // Enable cable loopback
    gmii_phy_setReg( phynum,  0, 0x0140 );       // Force 1000mbps, full duplex
    DM814x_usecDelay( 100000 );

    /* Wait for link */
    printf( "    Waiting for link...\n" );
    while( ( gmii_phy_getReg( phynum, 1 ) & 0x04 ) == 0 );
    printf( "    Link Detected\n" );
    printf ("Enabling the R-GMII interface on DM814X.\r\n");
    GMII_SEL = 0x0000030a;
    printf( "    In R-GMII mode\n" );
	
	EMAC_STAT_PORT_EN = 0x7;
	EMAC_P2_MAX_BLKS = 0x104;

   //gmii_phy_dumpRegs( );

    /* ---------------------------------------------------------------- *
     *  Init EMAC                                                       *
     * ---------------------------------------------------------------- */
    /* 1. Disable RX/TX interrupts */
    EMAC_EWRXEN = 0x00000000;
    EMAC_EWTXEN = 0x00000000;

    /* 2. Clear the MAC control, receive control, & transmit control */

    EMAC1_MACCONTROL = 0;
    EMAC_RXCONTROL = 0;
    EMAC_TXCONTROL = 0;

    /* 3. Initialize all 16 header descriptor pointers RXnHDP & TXnHDP to zero */
    EMAC_RX0HDP = 0;
	EMAC_RX1HDP = 0;
	EMAC_RX2HDP = 0;
	EMAC_RX3HDP = 0;
	EMAC_RX4HDP = 0;
	EMAC_RX5HDP = 0;
	EMAC_RX6HDP = 0;
	EMAC_RX7HDP = 0;

    EMAC_TX0HDP = 0;
	EMAC_TX1HDP = 0;
	EMAC_TX2HDP = 0;
	EMAC_TX3HDP = 0;
	EMAC_TX4HDP = 0;
	EMAC_TX5HDP = 0;
	EMAC_TX6HDP = 0;
	EMAC_TX7HDP = 0;

    /* 4. Clear all 36 statistics registers by writing 0 */
    pReg = &EMAC_RXGOODFRAMES;
    for ( i = 0 ; i < 36 ; i++ )
        *(pReg+i) = *(pReg+i);

    /* 8. Set RX buffer offset to 0.  Valid data always begins on the 1st byte */
    EMAC_RXBUFFEROFFSET = 0;

    /* 11. Set the appropriate configuration bits in MACCONTROL (do not set the GMIIEN bit yet). */
    EMAC1_MACCONTROL = 0
        | ( 0 << 9 )            // Round robin
        | ( 1 << 7 )            // Gigabit mode
        | ( 0 << 6 )            // TX pacing disabled
        | ( 0 << 5 )            // GMII RX & TX
        | ( 0 << 4 )            // TX flow disabled
        | ( 0 << 3 )            // RX flow disabled
        | ( 0 << 1 )          // Loopback disabled
        /*| ( 1 << 1 )*/            // Loopback enabled
        | ( 1 << 0 )           // full duplex
        | ( 1 << 22 ) 		// CEF
        | ( 1 << 23 );		// CSF


#if 0
	DM814x_usecDelay( 100000 );
	ale_entry[2] = 0x80000000; /* */
	ale_entry[1] = 0x10000000; /* Entry Type Unicast */
	ale_entry[0] = 0x00000000;
	ale_entry[1] |= 0xBB1300; 
	EMAC_ALE_TBLWRD2 = ale_entry[2];
	EMAC_ALE_TBLWRD1 = ale_entry[1];
	EMAC_ALE_TBLWRD0 = ale_entry[0];
#endif

	EMAC_ALE_PORTCTL0 |= 0x13;
	EMAC_ALE_PORTCTL1 |= 0x13;
	EMAC_ALE_PORTCTL2 |= 0x13;

	/* P2 TX Header Priority to Switch Priority Mapping */
	EMAC_P2_TX_PRI_MAP = 0x00;
	
    /* 12. Clear all unused channel interrupt bits */
    EMAC_RXINTMASKCLEAR = 0xFF;
    EMAC_TXINTMASKCLEAR = 0xFF;

    /* 13. Enable the RX & TX channel interrupt bits. */
    EMAC_RXINTMASKSET = 0xFF;
    EMAC_TXINTMASKSET = 0xFF;

    /* Enable Host Error and Statistics interrupts */
    EMAC_MACINTMASKSET = 0
        | ( 1 << 1 )            // Host Error interrupt mask
        | ( 1 << 0 );           // Statistics interrupt mask

    /* 14. Initialize the receive and transmit descriptor list queues. */

    /* 15. Prepare receive by writing a pointer to the head of the receive buffer descriptor list to RXnHDP. */
    EMAC1_MACSRCADDRLO = 0x03BB1300;   /* bytes 0, 1 */
    EMAC1_MACSRCADDRHI = 0xD967;       /* bytes 2-5 - channel 0 ??? */

    /* 16. Enable the RX & TX DMA controllers. Then set GMIIEN */
    EMAC_RXCONTROL = 1;
    EMAC_TXCONTROL = 1;

    EMAC1_MACCONTROL |= ( 1 << 5 );

    /* 17. Enable the device interrupt wrapper, emulation free mode */
    EMAC_EWRXEN      = 0x00000001; // Enable receive interrupts on channel 0
    EMAC_EWTXEN      = 0x00000001; // Enable transmit interrupts on channel 0


    return 0;
}

/* ------------------------------------------------------------------------ *
 *  verify_packet                                                           *
 * ------------------------------------------------------------------------ */
static INT16 verify_packet( EMAC_Desc* pDesc, UINT32 size, UINT32 flagCRC )
{
    INT16 i;
    UINT32 SizeCRC      = ( flagCRC ) ? size + 4 : size;
    UINT32 packet_flags = pDesc->PktFlgLen;

    /* We must own this packet */
    if ( packet_flags & EMAC_DSC_FLAG_OWNER )
        return 1;

    /* Must be SOP and EOP */
    if ( ( packet_flags & ( EMAC_DSC_FLAG_SOP | EMAC_DSC_FLAG_EOP ) )
                       != ( EMAC_DSC_FLAG_SOP | EMAC_DSC_FLAG_EOP ) )
        return 2;


    /* If flagCRC is set, it must have a CRC */
    if ( flagCRC )
        if ( ( packet_flags & EMAC_DSC_FLAG_PASSCRC ) != EMAC_DSC_FLAG_PASSCRC )
            return 3;

    /* Packet must be correct size */
    if ( ( packet_flags & 0xFFFF ) != SizeCRC )
        return 5;

    /* Offset must be zero, packet size unchanged */
    if ( pDesc->BufOffLen != SizeCRC )
        return 6;

    /* Validate the data */
    for ( i = 0 ; i < size ; i++ )
        if ( pDesc->pBuffer[i] != i )
            return 7;

    return 0;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  test_gmii_packet( )                                                     *
 *      EMAC tests to send data to external loopback cable.                 *
 *                                                                          *
 * ------------------------------------------------------------------------ */
INT16 test_gmii_packet( )
{
    INT32 i;
    INT16 errors = 0;
    UINT32 status;
    EMAC_Desc* pDesc;

    /* ---------------------------------------------------------------- *
     *  Setup EMAC for GMII mode                                        *
     * ---------------------------------------------------------------- */
  //emac_gmii_init( );

    memset( packet_buffer1, 0x00, 128 );
    memset( packet_buffer2, 0x00, 128 );
    
    /* ---------------------------------------------------------------- *
     *                                                                  *
     *  Setup RX packets                                                *
     *                                                                  *
     * ---------------------------------------------------------------- */
    pDesc            = pDescBase;
    pDesc->pNext     = pDescBase + 1;
    pDesc->pBuffer   = packet_buffer1;
    pDesc->BufOffLen = RX_BUF;
    pDesc->PktFlgLen = EMAC_DSC_FLAG_OWNER;

    pDesc            = pDescBase + 1;
    pDesc->pNext     = 0;
    pDesc->pBuffer   = packet_buffer2;
    pDesc->BufOffLen = RX_BUF;
    pDesc->PktFlgLen = EMAC_DSC_FLAG_OWNER;

    /*
     *  Start RX receiving
     */
    pDescRx = pDescBase;
    EMAC_RX0HDP = ( UINT32 )pDescRx;
    
    /* ---------------------------------------------------------------- *
     *                                                                  *
     *  Setup TX packets                                                *
     *      Send 2 copies of the same packet using different sizes      *
     *                                                                  *
     * ---------------------------------------------------------------- */
    for ( i = 0 ; i < TX_BUF ; i++ )
        packet_data[i] = i;

    pDesc            = pDescBase + 10;
    pDesc->pNext     = pDescBase + 11;
    pDesc->pBuffer   = packet_data;
    pDesc->BufOffLen = TX_BUF - 4; // 4 bytes for CRC
    pDesc->PktFlgLen = EMAC_DSC_FLAG_OWNER
                     | EMAC_DSC_FLAG_SOP
                     | EMAC_DSC_FLAG_EOP
                     | (TX_BUF - 4);

    pDesc            = pDescBase + 11;
    pDesc->pNext     = 0;
    pDesc->pBuffer   = packet_data;
    pDesc->BufOffLen = TX_BUF - 4; // 4 bytes for CRC
    pDesc->PktFlgLen = EMAC_DSC_FLAG_OWNER
                     | EMAC_DSC_FLAG_SOP
                     | EMAC_DSC_FLAG_EOP
                     | (TX_BUF - 4);

    /*
     *  Start TX sending
     */
    TxCount = 0;
    RxCount = 0;
    pDescTx = pDescBase + 10;
    EMAC_TX0HDP = ( UINT32 )pDescTx;
    
    DM814x_usecDelay( 100000 );

	/* ACK TX */
	status = EMAC_TX0CP;
    EMAC_TX0CP = status;
    
    DM814x_usecDelay( 100000 );
    /* ACK RX */
    status = EMAC_RX0CP;
    EMAC_RX0CP = status;
    
    /* ---------------------------------------------------------------- *
     *                                                                  *
     *  Normally there would be a interrupt to service the EMAC RX/TX   *
     *  transmission.  Instead a short pause and manually call the ISR  *
     *  Interrupt Service Routine to check on the status.               *
     *                                                                  *
     *  You can later add in the ISR and the code to support.           *
     *                                                                  *
     * ---------------------------------------------------------------- */

    /* Busy period */
    DM814x_usecDelay( 100000 );

    /* ---------------------------------------------------------------- *
     *  Check packet and results                                        *
     * ---------------------------------------------------------------- */

    if ( verify_packet( pDescBase, TX_BUF - 4, 1 ) )        // Verify Size + Contents
        errors++;

    if ( verify_packet( pDescBase + 1, TX_BUF - 4, 1 ) )	// Verify Size + Contents
        errors++;
#if 0
    if ( ( status = EMAC_FRAME64 ) != 2 )           // Check # of 64 byte frames
        errors++;
    EMAC_FRAME64 = status;

    if ( ( status = EMAC_FRAME128T255 ) != 2 )      // Check # of 128-255 byte frames
        errors++;
    EMAC_FRAME128T255 = status;

    if ( ( status = EMAC_RXGOODFRAMES ) != 2 )      // Check # of Good RX frames
        errors++;
    EMAC_RXGOODFRAMES = status;

    if ( ( status = EMAC_TXGOODFRAMES ) != 2 )      // Check # of Good TX frames
        errors++;
    EMAC_TXGOODFRAMES = status;
#endif
    return errors;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  emac_gmii_test( )                                                       *
 *                                                                          *
 * ------------------------------------------------------------------------ */
INT16 emac_rgmii_test()
{
	UINT16 phyaddr[32];
    INT16 errors = 0, num, i;

    /* Detect PHYs */
    num = gmii_phy_detect( phyaddr );
    num = num; /* for compiler warnings */
			
    /* Setup EMAC for GMII mode */
    emac_gmii_init( phyaddr[1] );  // Lower address is on the Netra base board

    /* Test EMAC in GMII mode */
    for ( i = 0 ; i < 10 ; i++ )
        errors += test_gmii_packet( );

	if (!errors)
	{
		printf("Packet verification passed!!\n");
	}
	else
	{
		printf("Packet verification failed!!\n");
	}
    return errors;
}
